classdef class_REVS_engine
	%UNTITLED6 Summary of this class goes here
	%   Detailed explanation goes here
	
	properties
		matrix_vintage = enum_matrix_vintage.present;
        name = '';
		combustion_type;										% Combustion type specified via enum_engine_combustion_type
		displacement_L;											% Engine displacement [liters]
		inertia_kgm2;											% Engine rotational inertia [kg*m^2]
		
		cold_correction =  0;                                   % UDDS bag 1 consumption correction factor, for MTE Matrix
		
		% 		accessories;											% Accessory losses - accessory class or structure containing losses
		fuel;													% Structure Containing map fuel properties
		
		%initial_speed_radps;
		
		full_throttle_speed_radps;								% Maximum torque curve speed breakpoints [rad/sec]
		full_throttle_torque_Nm;								% Maximum torque curve [Nm]
		
		naturally_aspirated_speed_radps;						% Maximum torque curve without boost speed breakpoints [rad/sec]
		naturally_aspirated_torque_Nm;							% Maximum torque curve without boost [Nm]
		
		closed_throttle_speed_radps;							% Closed throttle (Motoring) torque curve speed breakpoints [rad/sec]
		closed_throttle_torque_Nm ;								% Closed throttle (Motoring) torque curve [Nm]
		
		throttle_duration_sec = 0.2;							% Torque response rate limiter [sec]
		boost_time_constant_sec = 0.5;							% RC time constant associated with turbocharger boost response [sec]
		
		idle_speed_radps;										% Target idle speed Dynamic Lookup - can be set to scalar value [rad/sec]
		idle_control_Kp = 37.5;									% Idle Speed Control PID Proportional Term
		idle_control_Ki = 375;									% Idle Speed Control PID Integral Term
		idle_control_Kd = 0;									% Idle Speed Control PID Derivative Term
		idle_control_ramp_radps = 100 * convert.rpm2radps;      % Idle Speed Control Idle Ramp Speed [rad/s]
        idle_control_ramp_time_secs = 2;                        % Idle Speed Control Idle Ramp Time [sec]

		idle_control_activation_speed_radps = 1;				% Speed above which idle speed control PID activates [radians/sec]
		idle_control_deactivation_speed_radps = 0;				% Speed below which idle speed control deactivates ( engine stall ) [radians/sec]
		
		pedal_map_type;											% Generic or custom accelerator pedal to torque mapping
		pedal_map_speed_radps;										% Accelerator pedal to torque map speed breakpoints [radians/sec]
		pedal_map_pedal_norm;									% Accelerator pedal to torque map pedal breakpoints [0 - 1]
		pedal_map_Nm;											% Accelerator pedal to torque map [Nm]
		
		fuel_map_speed_radps;									% Fuel consumption map speed breakpoints [rad/sec]
		fuel_map_torque_Nm										% Fuel consumption map torque breakpoints [Nm]
		fuel_map_gps;											% Fuel consumption map [g/sec]
		
		fast_torque_fuel_adjust_norm;							% Portion of fuel cut associated with fast torque reduction [0 - 1] generally 0 for gasoline 1 or diesel
		
		refuel_profile_mult = [1.0 1.0];						% Defuel -> Refuel penalty fuel multiplier profile
		refuel_profile_time_secs = [0.0, 1.0];					% Defuel -> Refuel penalty fuel multiplier profile time [sec]
		refuel_limit_gps = 0.0;									% Defuel -> Refuel penalty additional fuel limit [grams/sec]
		refuel_min_defuel_duration_secs = inf; 					% Defuel -> Refuel penalty defueled duration before triggering penalty
		
		acceleration_penalty_gpradps2 = 0.0;					% Acceleration Penalty fuel rate [g/(rad/sec^2)]
		acceleration_penalty_squelch_gps = 0.0;					% Acceleration Penalty squelch limit [g/sec]
		
		decel_fuel_shutoff_disabled = false;					% Continue fueling engine when operating on closed throttle curve
		decel_fuel_shutoff_condition = '1';						% Conditions to allow decel fuel cutoff when operating on closed throttle curve [DYNAMIC EXPRESSION]
		
	end
	
	
	
	properties ( Dependent = true, SetAccess = immutable, Transient = true)
		
		full_throttle_power_W;
		
	end
	
	methods
		
		function obj = set.idle_speed_radps(obj,val)
			
			if isnumeric(val) && isscalar( val)				
				obj.idle_speed_radps.axis_1.signal = 'veh_spd_mps';
				obj.idle_speed_radps.axis_1.breakpoints = [0,1];
				obj.idle_speed_radps.table = [val, val];
			else
				obj.idle_speed_radps = val;
			end
						
		end
		
		function val = max_torque_Nm(obj)
			val = max( obj.full_throttle_torque_Nm );
		end
		
		function val = 	min_torque_Nm(obj)
			val = min( obj.closed_throttle_torque_Nm );
		end
		
		function val = get.full_throttle_power_W(obj)
			val = obj.full_throttle_speed_radps .* obj.full_throttle_torque_Nm;
		end
		
		function val = get.fast_torque_fuel_adjust_norm(obj)
			val = REVS_class_default( obj.fast_torque_fuel_adjust_norm, obj.combustion_type == enum_engine_combustion_type.compression_ignition );
		end
		
		
		function val = get.pedal_map_Nm(obj)
			if isempty(obj.pedal_map_type) || obj.pedal_map_type == enum_engine_pedal_map_type.lookup_table
				val = obj.pedal_map_Nm ;
			else
				
				zero_pedal_torque_Nm = zeros(length(obj.full_throttle_speed_radps),1);
				full_pedal_torque_Nm = obj.full_throttle_torque_Nm(:) - min(0, interp1(obj.closed_throttle_speed_radps(:), obj.closed_throttle_torque_Nm(:), obj.full_throttle_speed_radps(:),'linear','extrap'));
				
				obj.pedal_map_pedal_norm =  [0 1];
				%obj.pedal_map_speed_radps = obj.full_throttle_speed_radps;
				
				switch( obj.pedal_map_type)
					
					case  enum_engine_pedal_map_type.linear_WOT
						val = [zero_pedal_torque_Nm, full_pedal_torque_Nm]';
						
					case enum_engine_pedal_map_type.linear_max_engine_torque
						val = [zero_pedal_torque_Nm, max(full_pedal_torque_Nm)*ones(size(full_pedal_torque_Nm))]';
						
					case enum_engine_pedal_map_type.max_engine_power
						closed_throttle_power_W = obj.closed_throttle_speed_radps .* obj.closed_throttle_torque_Nm;
						val = [zero_pedal_torque_Nm, (obj.max_power_W - min(closed_throttle_power_W)) ./ max(min(obj.idle_speed_radps.table(:)), obj.full_throttle_speed_radps(:))]';
						
					case enum_engine_pedal_map_type.linear_enhanced_WOT
						val = [zero_pedal_torque_Nm, 1.1 * full_pedal_torque_Nm]';
						
					case enum_engine_pedal_map_type.linear_enhanced_max_engine_torque
						val = [zero_pedal_torque_Nm, 1.1 * max(full_pedal_torque_Nm)*ones(size(full_pedal_torque_Nm))]';
						
					otherwise
						val = [];
				end
			end
		end
		
		
		function val = get.pedal_map_pedal_norm(obj)
			
			if isempty(obj.pedal_map_type) || obj.pedal_map_type == enum_engine_pedal_map_type.lookup_table;
				val = obj.pedal_map_pedal_norm;
			else
				val =  [0 1];
			end
			% Handle undefined pedal map types
			
		end
		
		function val = get.pedal_map_speed_radps(obj)
			
			if isempty(obj.pedal_map_type) || obj.pedal_map_type == enum_engine_pedal_map_type.lookup_table ;
				val = obj.pedal_map_speed_radps;
			else
				val =  obj.full_throttle_speed_radps;
			end
			% Handle undefined pedal map types
			
		end
		
		function [trq, spd] = full_throttle_hires_torque( obj)
			spd = 0:0.1:obj.full_throttle_speed_radps(end);
			trq = interp1( obj.full_throttle_speed_radps, obj.full_throttle_torque_Nm, spd );
		end
		
		
		function val = max_power_W(obj)
			[trq, spd] = obj.full_throttle_hires_torque;
			val = max( spd .* trq );
		end
		
		function val = max_power_max_radps(obj)
			[trq, spd] = obj.full_throttle_hires_torque;
			
			power_W = spd .* trq ;
			idx =  find( power_W > max(power_W) * 0.98,1,'last');
			val = spd(idx);
		end
		
		function val = max_power_max_rpm(obj)
			val = obj.max_power_max_radps .* convert.radps2rpm;
		end
		
		function val = max_power_min_radps(obj)
			[trq, spd] = obj.full_throttle_hires_torque;
			
			power_W = spd .* trq ;
			idx =  find( power_W > max(power_W) * 0.98,1,'first');
			val = spd(idx);
		end
		
		function val = max_power_min_rpm(obj)
			val = obj.max_power_min_radps .* convert.radps2rpm;
		end
		
		function val = max_power_avg_radps(obj)
			[trq, spd] = obj.full_throttle_hires_torque;
			power_W = spd .* trq;
			
			idx1 =  find( power_W > max(power_W) * 0.98,1,'last');
			idx2 =  find( power_W > max(power_W) * 0.98,1,'first');
			val = mean(spd([idx1,idx2]));
		end
		
		function val = max_power_avg_rpm(obj)
			val = obj.max_power_avg_radps .* convert.radps2rpm;
		end
		
		function val = max_test_speed_radps(obj)
			[trq, spd] = obj.full_throttle_hires_torque;
			power_W = spd .* trq ;
			
			% calculations following 1065.510 to find Maximum Test Speed (MTS)
			P_normi                    = power_W ./ obj.max_power_W;
			f_normi                    = spd ./ obj.max_power_avg_radps;
			sum_squares                = f_normi.^2 + P_normi.^2;
			max_test_speed_min_radps   = spd( find(sum_squares >= 0.98 * max(sum_squares), 1, 'first') );
			max_test_speed_radps       = spd( find(sum_squares >= 0.98 * max(sum_squares), 1, 'last') );
			val                        = mean( [max_test_speed_radps max_test_speed_min_radps] );
		end
		
		function val = max_test_speed_rpm(obj)
			val = obj.max_test_speed_radps * convert.radps2rpm;
		end
				
	end
	
end

